<?php
// includes/finance/quotes/finance_quotes_import.php

header('Content-Type: application/json');

$tenant = $_REQUEST['tenant'] ?? '';

if ($tenant) {
    if (file_exists('../../../../' . $tenant . '/config.php')) {
        require_once('../../../../' . $tenant . '/config.php');
    }
}
if (!isset($GLOBALS['con']) && file_exists('/config.php')) {
    require_once('/config.php');
}

if (!isset($GLOBALS['con'])) {
    echo json_encode(['status' => 'error', 'message' => 'Database connection not available.']);
    exit;
}

$func = $_REQUEST['FUNCTION'] ?? '';

switch ($func) {
    case 'importItemsFromCsv':
        importItemsFromCsv();
        break;

    default:
        echo json_encode(['status' => 'error', 'message' => 'Invalid function.']);
        break;
}

/**
 * Import items from a CSV matching the asset import template.
 * Only the "Description" column is required.
 * Each distinct matching description is grouped and looked up in dd_plant.
 */
function importItemsFromCsv(): void
{
    $con = $GLOBALS['con'];

    if (empty($_FILES['csv_file']) || !isset($_FILES['csv_file']['tmp_name'])) {
        echo json_encode(['status' => 'error', 'message' => 'No CSV file uploaded.']);
        return;
    }

    if ($_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
        echo json_encode(['status' => 'error', 'message' => 'Upload error: ' . (int)$_FILES['csv_file']['error']]);
        return;
    }

    $tmpName = $_FILES['csv_file']['tmp_name'];

    if (!is_readable($tmpName)) {
        echo json_encode(['status' => 'error', 'message' => 'Unable to read uploaded CSV file.']);
        return;
    }

    $handle = fopen($tmpName, 'r');
    if (!$handle) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to open CSV file.']);
        return;
    }

    // Header row
    $header = fgetcsv($handle, 0, ',');
    if ($header === false) {
        fclose($handle);
        echo json_encode(['status' => 'error', 'message' => 'CSV appears to be empty.']);
        return;
    }

    // Locate the "Description" column (case-insensitive)
    $descIndex = null;
    foreach ($header as $idx => $col) {
        if (strcasecmp(trim((string)$col), 'description') === 0) {
            $descIndex = $idx;
            break;
        }
    }

    if ($descIndex === null) {
        fclose($handle);
        echo json_encode([
            'status'  => 'error',
            'message' => 'CSV does not contain a "Description" column.'
        ]);
        return;
    }

    // Aggregate counts by description text
    $buckets = []; // key => ['raw_desc' => string, 'count' => int]

    while (($row = fgetcsv($handle, 0, ',')) !== false) {
        if (!isset($row[$descIndex])) {
            continue;
        }
        $desc = trim((string)$row[$descIndex]);
        if ($desc === '') {
            continue;
        }

        $key = mb_strtolower($desc, 'UTF-8');
        if (!isset($buckets[$key])) {
            $buckets[$key] = [
                'raw_desc' => $desc,
                'count'    => 0,
            ];
        }
        $buckets[$key]['count']++;
    }

    fclose($handle);

    if (!$buckets) {
        echo json_encode([
            'status'  => 'success',
            'items'   => [],
            'message' => 'No description values found in CSV.'
        ]);
        return;
    }

    // Prepare dd_plant lookup by description (case-insensitive, trimmed)
    $sql = "SELECT code, description, units 
            FROM dd_plant 
            WHERE LOWER(TRIM(description)) = ? 
            LIMIT 1";

    $stmt = $con->prepare($sql);
    if (!$stmt) {
        echo json_encode(['status' => 'error', 'message' => 'Failed to prepare dd_plant lookup.']);
        return;
    }

    $items = [];
    $unmatched = [];
    foreach ($buckets as $key => $bucket) {
        $lookup = $key; // already lowercased

        $stmt->bind_param('s', $lookup);
        $stmt->execute();
        $res = $stmt->get_result();

        $plant = $res->fetch_assoc();
        $res->free();

        if (!$plant) {
            // Return as unmatched so UI can add it to the table for manual completion
            $unmatched[] = [
                'item_type'   => 'inspection',
                'description' => $bucket['raw_desc'],
                'qty'         => (int)$bucket['count'],
                'unit_price'  => 0,
                'net'         => 0,
                'vat'         => 0,
                'gross'       => 0,
                'unmatched'   => 1
            ];
            continue;
        }


        $count        = (int)$bucket['count'];
        $unitsPerItem = isset($plant['units']) ? (float)$plant['units'] : 0.0;

        $lineDesc = $plant['description'] ?? $bucket['raw_desc'];

        if (!empty($plant['code'])) {
            $lineDesc .= ' (Code: ' . $plant['code'] . ')';
        }
        if ($unitsPerItem > 0) {
            $lineDesc .= ' – ' . $unitsPerItem . ' IU per item';
        }

        // We mirror the buildItemsFromClient pattern:
        // - item_type = ui_item
        // - qty = number of items (asset count)
        // - unit_price left as 0; IU rate is controlled in the IU block
        $items[] = [
            'item_type'      => 'inspection',
            'description'    => $lineDesc,
            'qty'            => $count,
            'unit_price'     => 0,
            'net'            => 0,
            'vat'            => 0,
            'gross'          => 0,
            'units_per_item' => $unitsPerItem,
            'item_count'     => $count,
        ];
    }

    $stmt->close();

    if (!$items) {
        echo json_encode([
            'status'  => 'success',
            'items'   => [],
            'message' => 'No dd_plant matches were found for CSV descriptions.'
        ]);
        return;
    }

    echo json_encode([
        'status'    => 'success',
        'items'     => $items,
        'unmatched' => $unmatched,
    ]);
}
