<?php
// elec_new_report.php

// Extract board data from the pipe-delimited string in $_REQUEST['data']
$d = explode("|", $_REQUEST['data']);
if (!isset($d[0])) {
    die("Error: Board ID is missing.");
}
$board_id     = intval($d[0]);
$clientnumber = intval($d[1]);
$locnumber    = intval($d[2]);

// Fetch Board Details
function get_board_details($board_id) {
    global $con;
    $query = "SELECT b.*, l.locname, l.postcode
              FROM elec_board b
              LEFT JOIN locations l ON b.locnumber = l.locnumber
              WHERE b.board_id = ?";
    $stmt = $con->prepare($query);
    if (!$stmt) {
        die("Prepare failed: " . $con->error);
    }
    $stmt->bind_param("i", $board_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $stmt->close();

    if ($row) {
        $row['clientname'] = !empty($row['clientnumber']) ? getClientName($row['clientnumber']) : 'Unknown';
        $row['locname'] = !empty($row['locname']) ? $row['locname'].' ('.$row['postcode'].')' : 'Unknown';
        // Remove these fields from the output
        unset($row['locnumber'], $row['clientnumber'], $row['postcode'], $row['board_id']);
        if (!empty($row['installation_date'])) {
            $dt = DateTime::createFromFormat('Y-m-d', $row['installation_date']);
            if ($dt !== false) {
                $row['installation_date'] = $dt->format('d-m-Y');
            } else {
                $timestamp = strtotime($row['installation_date']);
                if ($timestamp !== false) {
                    $row['installation_date'] = date('d-m-Y', $timestamp);
                }
            }
        }
        // Convert any 1/0 values to Yes/No for display
        foreach ($row as $key => $value) {
            if ($value === "1" || $value === 1) {
                $row[$key] = "Yes";
            } elseif ($value === "0" || $value === 0) {
                $row[$key] = "No";
            }
        }
        return $row;
    }
    return [];
}

// Get circuits for the board
$query = "SELECT circuit_id, circuit_no, circuit_description FROM elec_circuit WHERE board_id = ? ORDER BY circuit_no ASC";
$stmt = $con->prepare($query);
$stmt->bind_param("i", $board_id);
$stmt->execute();
$result = $stmt->get_result();
$circuitsCount = $result->num_rows;
$circuitsInfo = [];
while ($row = $result->fetch_assoc()) {
    $circuitsInfo[$row['circuit_id']] = $row;
}
$stmt->close();

// Check for an incomplete report for this board
function check_incomplete_report($board_id) {
    global $con;
    $query = "SELECT master_report_id FROM elec_report WHERE board_id = ? AND status = 'active'";
    $stmt = $con->prepare($query);
    $stmt->bind_param("i", $board_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $data = $result->fetch_assoc();
    $stmt->close();
    return $data;
}

$board_details = get_board_details($board_id);
// Check if an incomplete report exists for this board
$incomplete_report = check_incomplete_report($board_id);
if (!empty($incomplete_report['master_report_id'])) {
    $master_report_id = $incomplete_report['master_report_id'];
    echo "<script>
        let MASTER_REPORT_ID = '$master_report_id'; // Use the existing master_report_id
        let is_update = true; // Set the update flag to true
    </script>";
} else {
    echo "<script>
        let MASTER_REPORT_ID = null; // No existing report found
        let is_update = false; // No update needed
    </script>";
}

// Check for an existing report with the master_report_id
if (!empty($incomplete_report['master_report_id'])) {
    $query = "SELECT installation_details FROM elec_report WHERE master_report_id = ?";
    $stmt = $con->prepare($query);
    $stmt->bind_param("i", $incomplete_report['master_report_id']);
    $stmt->execute();
    $result = $stmt->get_result();
    $reportData = $result->fetch_assoc();
    $stmt->close();
    // Update REPORT_ID to master_report_id
    $REPORT_ID = $incomplete_report['master_report_id'];
}
$installation_data = !empty($reportData['installation_details']) ? json_decode($reportData['installation_details'], true) : [];

// Surveyor functions
function getSurveyorNameForm($esname){
    $stmt = $GLOBALS['con']->prepare("SELECT `surname`,`forename` FROM `surveyor` WHERE `id`=?") or die(mysqli_error($GLOBALS['con']));
    $stmt->bind_param('i', $esname);
    $stmt->execute();
    $rc = $stmt->get_result();
    $stmt->close();
    $cd = mysqli_fetch_array($rc);
    return $cd['forename']." ".$cd['surname'];
}

function listSurveyor($c, $n, $t){
    if(!empty($_SESSION['contractorNumber'])){
        $filter = "AND `contractor`='".$_SESSION['contractorNumber']."'";
    }
    switch($t){
        case "select":
            $ca = array();
            $stmt = $GLOBALS['con']->prepare("SELECT `id`,`esname`,`forename`,`surname` FROM `surveyor` WHERE `statuslive`='1' ORDER BY `forename` ASC") or die(mysqli_error($GLOBALS['con']));
            $stmt->execute();
            $cc = $stmt->get_result();
            $stmt->close();
            while($pd = mysqli_fetch_array($cc)) { $ca[] = $pd; }
            $ret = '<select class="form-control form-control-sm" data-mdb-select-init data-mdb-filter="true" id="'.$n.'" name="'.$n.'">';
            $ret .= '<option value="NULL">Select primary Surveyor</option>';
            foreach($ca as $px){
                $name = $px['forename'].' '.$px['surname'];
                $s = ($c == $px['id']) ? "SELECTED" : "";
                $ret .= '<option value="'.$px['id'].'" '.$s.'>'.$name.'</option>';
            }
            $ret .= '</select>';
            return $ret;
        break;
        case "name":
            $ca = array();
            $stmt = $GLOBALS['con']->prepare("SELECT `id`,`esname`,`forename`,`surname` FROM `surveyor` WHERE `statuslive`='1' ORDER BY `forename` ASC") or die(mysqli_error($GLOBALS['con']));
            $stmt->execute();
            $cc = $stmt->get_result();
            $stmt->close();
            while($pd = mysqli_fetch_array($cc)) { $ca[] = $pd; }
            $ret = '<select class="form-control form-control-sm" id="'.$n.'" name="'.$n.'">';
            $ret .= '<option value="NULL">Select Surveyor</option>';
            foreach($ca as $px){
                $name = $px['forename'].' '.$px['surname'];
                $s = ($c == $name) ? "SELECTED" : "";
                $ret .= '<option value="'.$name.'" '.$s.'>'.$name.'</option>';
            }
            $ret .= '</select>';
            return $ret;
        break;
        case "ajax":
            $ca = array();
            $stmt = $GLOBALS['con']->prepare("SELECT `id`,`esname`,`forename`,`surname` FROM `surveyor` WHERE `statuslive`='1' ORDER BY `forename` ASC") or die(mysqli_error($GLOBALS['con']));
            $stmt->execute();
            $cc = $stmt->get_result();
            $stmt->close();
            $cnt = mysqli_num_rows($cc);
            while($pd = mysqli_fetch_array($cc)) { $ca[] = $pd; }
            $co = 0;
            $ret = 'var clientList = [';
            foreach($ca as $px){
                $name = $px['forename'].' '.$px['surname'];
                $name = str_replace('"','',$name);
                $name = str_replace('\&','and',$name);
                $co++;
                $s = ($co == $cnt) ? "" : ",";
                $ret .= '{"value":"'.$name.'","name":"'.$px['id'].'"}'.$s."\n";
            }
            $ret .= '];';
            return $ret;
        break;
    }
}

// Build board list using both clientnumber and locnumber
$locationBoards = [];
$query = "SELECT board_id, board_name FROM elec_board WHERE clientnumber = ? AND locnumber = ?";
$stmt = $con->prepare($query);
$stmt->bind_param("ii", $clientnumber, $locnumber);
$stmt->execute();
$result = $stmt->get_result();
while ($row = $result->fetch_assoc()){
    $locationBoards[] = $row;
}
$stmt->close();
?>


<!-- Pass the full board list to JavaScript and as a hidden field for saves -->
<script>
const ALL_BOARD_LIST = <?= json_encode($locationBoards) ?>;
var locationBoards = ALL_BOARD_LIST;
</script>
<input type="hidden" id="board_list" name="board_list" value='<?= htmlspecialchars(json_encode($locationBoards, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES), ENT_QUOTES, 'UTF-8') ?>'>

<input type="text" value="<?= $incomplete_report['master_report_id']; ?>">
<style>
#toggleBoardDetails {
    position: fixed;
    top: 48px;
    left: 50%;
    transform: translateX(-50%);
    z-index: 1050;
}

#boardDetailsContainer {
    position: fixed;
    top: 48px;
    left: 50%;
    transform: translateX(-50%);
    width: 90%;
    max-width: 800px;
    z-index: 1040;
}
</style>

<button id="toggleBoardDetails" class="btn btn-secondary" type="button" data-mdb-toggle="collapse" data-mdb-target="#boardDetailsContainer" aria-expanded="false" aria-controls="boardDetailsContainer">
    Board Details
</button>

<?php
$board_details = get_board_details($board_id);
if (!empty($board_details)) {
    echo "<div class='collapse bg-info text-light p-3 border shadow-4-strong rounded-bottom' id='boardDetailsContainer'>";
    echo "<h2>Board Details</h2>";
    echo "<div class='row bg-dark text-light rounded'>";
    foreach ($board_details as $field => $value) {
        $formattedField = ucwords(str_replace("_", " ", $field));
        echo "<div class='col-md-3 mb-3'><div class='p-2'><strong style='color:darkgrey'>" . htmlspecialchars($formattedField) . ":</strong><br>" . htmlspecialchars($value) . "</div></div>";
    }
    echo "</div>";
    echo "</div>";
} else {
    echo "<p>No board details found.</p>";
}
?>

<link rel="stylesheet" href="<?= ROOT_URL ?>/src/uploader.css?v=5.0.3">
<div class="container-fluid">
    <div class="row mb-4 pt-3">
        <div class="col-md-8">
            <h3 class="text-dark">
                <?= $incomplete_report 
                    ? '<i class="fad fa-regular fa-arrow-right-to-bracket"></i> Resume Report' 
                    : '<i class="fad fa-regular fa-square-plus"></i> Create New Report' ?>
            </h3>
            <p>
                Board: <?= htmlspecialchars($board_details['board_name']) ?>
                Location: <?= htmlspecialchars($board_details['locname']) ?>
            </p>
        </div>
        <div class="col-md-4 text-md-end text-center">
            <button class="btn btn-secondary mt-3" onclick="window.history.back()">
                <i class="fas fa-arrow-left"></i> Cancel
            </button>
        </div>
    </div>

    <div class="row">
        <div class="col-12 col-md-2">
            <h5>Report Stages</h5>
            <ul class="list-group" id="reportChecklist">
                <li class="list-group-item mb-2 p-2 border-top bg-info text-light" onclick="navigateStage(1)" style="cursor: pointer; font-size:12px">
                    1. Details of Installation
                    <span class="float-end" id="stage1-status">
                        <i class="fad fa-spinner-third text-info fa-spin"></i>
                    </span>
                </li>
                <li class="list-group-item mb-2 p-2 border-top bg-info text-light" onclick="navigateStage(2)" style="cursor: pointer; font-size:12px">
                    2. Schedule of Inspection
                    <span class="float-end" id="stage2-status">
                        <i class="fad fa-spinner-third text-info fa-spin"></i>
                    </span>
                </li>
                <li class="list-group-item mb-2 p-2 border-top bg-info text-light" onclick="navigateStage(3)" style="cursor: pointer; font-size:12px">
                    3. Condition Report
                    <span class="float-end" id="stage3-status">
                        <i class="fad fa-spinner-third text-info fa-spin"></i>
                    </span>
                </li>

            </ul>
            <h5>Inspection and Test Results (Per Board)</h5>
            <ul class="list-group" id="locationBoardsList">
                <?php foreach ($locationBoards as $board): ?>
                <li class="list-group-item mb-2 p-2 border-top bg-info text-light" onclick="navigateStage4(<?= $board['board_id'] ?>)" style="cursor: pointer; font-size:12px">
                    Board: <?= htmlspecialchars($board['board_name']) ?>
                    <span class="float-end" id="stage4-status-<?= $board['board_id'] ?>">
                        <i class="fad fa-spinner-third text-info fa-spin"></i>
                    </span>
                </li>
                <?php endforeach; ?>
            </ul>
        </div>
        <div class="col-12 col-md-10">
            <div id="stageContent">
                <div id="landingMessage" class="d-flex flex-column align-items-center justify-content-center my-5">
                    <span class="fa-stack fa-4x">
                        <i class="fad fa-square fa-stack-2x text-dark"></i>
                        <i class="fad fa-bolt fa-stack-1x text-light fa-inverse"></i>
                    </span>
                    <h2 class="mt-3 text-primary">Electrical Inspection Report</h2>
                    <p class="lead text-muted">Select a stage from the checklist to begin your inspection.</p>
                </div>
            </div>
        </div>
    </div>
</div>
<script src="https://cdnjs.cloudflare.com/ajax/libs/mdb-ui-kit/6.2.0/mdb.min.js"></script>
<script src="<?= ROOT_URL ?>/src/uploader.js"></script>
<script>
// Expose locationBoards array to JavaScript.
var locationBoards = <?= json_encode($locationBoards) ?>;

// Initialize MDB collapse for board details.
document.addEventListener('DOMContentLoaded', function() {
    if (typeof mdb !== 'undefined') {
        const element = document.querySelector('.collapse');
        new mdb.Collapse(element);
    } else {
        console.error("mdb is not defined. Please check that your MDBootstrap script is loaded correctly.");
    }
});

function alertMessage(title, message, target) {
    var dest = `<p>${message}</p>`;
    loadModalContent(dest, "confirmBox", "modal-md", 1060, "true", title, "cancelConfirm", target);
}

function toggleImageUploaders() {
    var reportId = document.getElementById("master_report_id").value.trim();
    var uploaderContainers = document.querySelectorAll('.imageContainer');
    uploaderContainers.forEach(function(container) {
        var existingMsg = container.parentNode.querySelector('.uploader-message');
        if (!reportId) {
            container.style.display = "none";
            if (!existingMsg) {
                var message = document.createElement("div");
                message.className = "uploader-message";
                message.style.textAlign = "center";
                message.style.padding = "10px";
                message.style.fontWeight = "bold";
                message.textContent = "Image uploaders available after initial save";
                container.parentNode.insertBefore(message, container);
            }
        } else {
            container.style.display = "block";
            if (existingMsg) {
                existingMsg.remove();
            }
        }
    });
}

// Global navigation for stages 1-3 – now including board_list if available
function navigateStage(stage) {
    // Ensure ALL_BOARD_LIST is defined and non-empty before encoding
    var boardListParam = "";
    if (typeof ALL_BOARD_LIST !== "undefined" && ALL_BOARD_LIST.length > 0) {
        boardListParam = encodeURIComponent(JSON.stringify(ALL_BOARD_LIST));
    }
    // Build URL; note we pass locnumber and tenant; board_list is appended if available
    var url = `${ROOT_URL}/includes/elec/load_stage.php?stage=${stage}&locnumber=<?= $locnumber ?>&tenant=${TENANT_URL}`;
    if (boardListParam !== "") {
        url += `&board_list=${boardListParam}`;
    }

    loadContentAndExecuteScripts(url, "stageContent", "", function() {
        fetch(`${ROOT_URL}/includes/elec/get_report_status.php?master_report_id=${MASTER_REPORT_ID}&tenant=${TENANT_URL}`)

            .then(response => response.json())
            .then(response => {
                if (response.success) {
                    const data = response.data;
                    if (stage === 3) {
                        toggleImageUploaders();
                    }
                    var markCompleteBtn = document.getElementById('markCompleteBtn');
                    if (markCompleteBtn) {
                        markCompleteBtn.style.display = (data.stage_status[stage] === 'complete') ? 'none' : 'block';
                    }
                }
            })
            .catch(error => console.error('Error fetching report data:', error));
    });
}


// Navigation for Stage 4 (board‑specific)
function navigateStage4(boardId) {
    var boardListParam = encodeURIComponent(JSON.stringify(ALL_BOARD_LIST));
    var is_update = (MASTER_REPORT_ID !== null) ? "true" : "false"; // Check if it's update mode
    var url = `${ROOT_URL}/includes/elec/load_stage.php?stage=4&tenant=${TENANT_URL}&board_id=${boardId}&is_update=${is_update}&master_report_id=${MASTER_REPORT_ID}&board_list=${boardListParam}`;
    loadContentAndExecuteScripts(url, "stageContent", "", function() {
        fetch(`${ROOT_URL}/includes/elec/get_report_status.php?report_id=${REPORT_ID}&tenant=${TENANT_URL}`)
            .then(response => response.json())
            .then(response => {
                if (response.success) {
                    var status = (response.data.circuit_results && response.data.circuit_results[boardId]) ? response.data.circuit_results[boardId] : "empty";
                    updateStage4Status(boardId, status);
                }
            })
            .catch(error => console.error('Error fetching report data:', error));
    });
}


function checkStageCompletion() {

    if (!MASTER_REPORT_ID) {
        updateStageStatus(1, "empty");
        updateStageStatus(2, "empty");
        updateStageStatus(3, "empty");
        updateStageStatus(4, {}); // empty object for stage 4
        return;
    }
    fetch(`${ROOT_URL}/includes/elec/get_report_status.php?master_report_id=${MASTER_REPORT_ID}&tenant=${TENANT_URL}`)
        .then(response => response.json())
        .then(response => {
            if (response.success) {
                const data = response.data;
                updateStageStatus(1, data.stage_status[1] || (Object.keys(data.installation_details).length > 0 ? 'ongoing' : 'empty'));
                updateStageStatus(2, data.stage_status[2] || (Object.keys(data.schedule_of_inspection).length > 0 ? 'ongoing' : 'empty'));
                updateStageStatus(3, data.stage_status[3] || (Object.keys(data.condition_schedule).length > 0 ? 'ongoing' : 'empty'));
                updateStageStatus(4, data.circuit_results || {});
            } else {
                console.error('Error fetching report data:', response.message);
            }
        })
        .catch(error => console.error('Error fetching report data:', error));
}

function updateStageStatus(stage, status) {
    if (stage === 4) {
        if (typeof status === 'object' && status !== null) {
            locationBoards.forEach(function(board) {
                var boardStatus = status[board.board_id] || 'empty';
                updateStage4Status(board.board_id, boardStatus);
            });
        } else {
            locationBoards.forEach(function(board) {
                updateStage4Status(board.board_id, 'empty');
            });
        }
    } else {
        const statusElem = document.getElementById("stage" + stage + "-status");
        if (statusElem) {
            let iconHTML = '';
            switch (status) {
                case 'pending':
                    iconHTML = '<i class="fad fa-spinner-third text-info fa-spin"></i> Ongoing';
                    break;
                case 'complete':
                    iconHTML = '<i class="fad fa-check-circle text-success"></i> Complete';
                    break;
                case 'empty':
                    iconHTML = '<i class="fad fa-minus-circle text-danger"></i> Not started';
                    break;
            }
            statusElem.innerHTML = iconHTML;
        }
    }
    checkAllStagesComplete();
}

function updateStage4Status(boardId, status) {
    const statusElem = document.getElementById("stage4-status-" + boardId);
    if (statusElem) {
        let iconHTML = '';
        switch (status) {
            case 'pending':
                iconHTML = '<i class="fad fa-spinner-third text-info fa-spin"></i> Ongoing';
                break;
            case 'complete':
                iconHTML = '<i class="fad fa-check-circle text-success"></i> Complete';
                break;
            case 'empty':
                iconHTML = '<i class="fad fa-minus-circle text-danger"></i> Not started';
                break;
        }
        statusElem.innerHTML = iconHTML;
    }
    checkAllStagesComplete();
}

function checkAllStagesComplete() {
    const stage1HTML = document.getElementById("stage1-status").innerHTML.toLowerCase();
    const stage2HTML = document.getElementById("stage2-status").innerHTML.toLowerCase();
    const stage3HTML = document.getElementById("stage3-status").innerHTML.toLowerCase();
    let allStage4Complete = true;
    locationBoards.forEach(function(board) {
        const el = document.getElementById("stage4-status-" + board.board_id);
        if (!el || !el.innerHTML.toLowerCase().includes("complete")) {
            allStage4Complete = false;
        }
    });
    const allComplete = stage1HTML.includes("complete") &&
        stage2HTML.includes("complete") &&
        stage3HTML.includes("complete") &&
        allStage4Complete;
    const checklist = document.getElementById("reportChecklist");
    if (allComplete) {
        if (!document.getElementById("finaliseReportBtn")) {
            const btn = document.createElement("button");
            btn.id = "finaliseReportBtn";
            btn.className = "btn btn-success mt-2";
            btn.innerHTML = '<i class="fad fa-check-double me-1"></i> Finalise Report';
            checklist.appendChild(btn);
            btn.addEventListener("click", finaliseReport);
        }
        if (!document.getElementById("cancelReportBtn")) {
            const btncancel = document.createElement("button");
            btncancel.id = "cancelReportBtn";
            btncancel.className = "btn btn-danger mt-5";
            btncancel.innerHTML = '<i class="fad fa-trash-xmark me-1"></i> Cancel Report';
            checklist.appendChild(btncancel);
            btncancel.addEventListener("click", cancelReport);
        }
    } else {
        const existingBtn = document.getElementById("finaliseReportBtn");
        const cancelRptBtn = document.getElementById("cancelReportBtn");
        if (existingBtn) {
            existingBtn.remove();
        }
        if (cancelRptBtn) {
            cancelRptBtn.remove();
        }
    }
}

function updateStageState(stage, status) {
    let formData = new URLSearchParams();
    formData.append("master_report_id", MASTER_REPORT_ID || "");
    formData.append("stage", stage);
    formData.append("status", status);
    formData.append("tenant", TENANT_URL);
    formData.append("is_update", is_update);
    fetch(`${ROOT_URL}/includes/elec/update_stage_status.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: formData.toString()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateStageStatus(stage, 'pending');
                checkStageCompletion();
                checkAllStagesComplete();
            } else {
                console.error('Error updating stage:', data.message);
            }
        })
        .catch(error => console.error('Error updating stage:', error));
}

document.addEventListener("DOMContentLoaded", () => {
    checkStageCompletion();
    updateInstrumentRowIcons();
});

// After saving the report and getting the master_report_id:
function saveReportStage(stage) {
    let formData = new FormData();
    formData.append("stage", `save_stage_${stage}`);
    formData.append("board_list", document.getElementById("board_list").value);
    formData.append("tenant", TENANT_URL);
    formData.append("clientnumber", 1); // Replace 1 with your dynamic client number if needed.
    formData.append("is_update", is_update); // Pass is_update to indicate update mode
    formData.append("master_report_id", MASTER_REPORT_ID); // Pass the master_report_id (if available)

    let formElement;
    switch (stage) {
        case 1:
            formElement = document.getElementById("installationDetailsForm");
            break;
        case 2:
            formElement = document.getElementById("scheduleOfInspectionForm");
            break;
        case 3:
            formElement = document.getElementById("conditionReportForm");
            break;
        case 4:
            formElement = document.getElementById("circuitInspectionForm");
            break;
        default:
            console.error("Invalid stage: " + stage);
            return;
    }

    // Append all form data
    let inputs = formElement.querySelectorAll("input, select, textarea");
    inputs.forEach(input => {
        if (input.type === "radio") {
            if (input.checked) {
                formData.append(input.name, input.value);
            }
        } else if (input.type === "checkbox") {
            formData.append(input.name, input.checked ? 1 : 0);
        } else {
            formData.append(input.name, input.value);
        }
    });

    fetch(`${ROOT_URL}/includes/elec/load_stage.php?tenant=${TENANT_URL}`, {
            method: "POST",
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (data.master_report_id) {
                    // Update MASTER_REPORT_ID if a new report is created or existing report is updated
                    MASTER_REPORT_ID = data.master_report_id;

                    // Set is_update flag to true since we are now updating the report
                    is_update = true;

                    // Log the updated values (for debugging purposes)
                    console.log("Updated MASTER_REPORT_ID:", MASTER_REPORT_ID);
                    console.log("Updated is_update:", is_update);

                    // Optionally, trigger actions here after the update
                    handleReportUpdate();
                }
                killModal('confirmBox');
                showResponse("success", data.message);
                updateStageStatus(stage, 'pending');
                updateStageState(stage, "pending");
                checkStageCompletion();
                document.getElementById("stageContent").innerHTML = `
                    <div id="landingMessage" class="d-flex flex-column align-items-center justify-content-center my-5">
                        <span class="fa-stack fa-4x">
                            <i class="fad fa-square fa-stack-2x text-primary"></i>
                            <i class="fad fa-bolt fa-stack-1x fa-inverse"></i>
                        </span>
                        <h2 class="mt-3 text-primary">Electrical Inspection Report</h2>
                        <p class="lead text-muted">Select a stage from the checklist to begin your inspection.</p>
                    </div>`;
            } else {
                showResponse("danger", data.message);
            }
        })
        .catch(error => {
            console.error("Save error:", error);
            showResponse("danger", "Failed to save data.");
        });
}

function saveCircuitStage4(board_id) {
    let formData = new FormData();
    formData.append("stage", "save_stage_4");
    formData.append("board_list", document.getElementById("board_list").value);
    formData.append("tenant", TENANT_URL);
    formData.append("clientnumber", 1); // Replace 1 with your dynamic client number if needed.
    formData.append("is_update", is_update); // Pass is_update to indicate update mode
    formData.append("master_report_id", MASTER_REPORT_ID); // Pass the master_report_id (if available)
    formData.append("board_id", board_id); // Pass the specific board_id for this stage

    // Get the form element for this stage
    let formElement = document.getElementById("circuitInspectionForm");

    // Append all form data (input values) from the form
    let inputs = formElement.querySelectorAll("input, select, textarea");
    inputs.forEach(input => {
        if (input.type === "radio") {
            if (input.checked) {
                formData.append(input.name, input.value);
            }
        } else if (input.type === "checkbox") {
            formData.append(input.name, input.checked ? 1 : 0);
        } else {
            formData.append(input.name, input.value);
        }
    });

    // Send the form data to the server
    fetch(`${ROOT_URL}/includes/elec/load_stage.php?tenant=${TENANT_URL}`, {
            method: "POST",
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (data.master_report_id) {
                    // Update MASTER_REPORT_ID if a new report is created or existing report is updated
                    MASTER_REPORT_ID = data.master_report_id;

                    // Set is_update flag to true since we are now updating the report
                    is_update = true;

                    // Log the updated values (for debugging purposes)
                    console.log("Updated MASTER_REPORT_ID:", MASTER_REPORT_ID);
                    console.log("Updated is_update:", is_update);

                    // Optionally, trigger actions here after the update
                    handleReportUpdate();
                }

                killModal('confirmBox');
                showResponse("success", data.message);
                updateStageStatus(4, 'pending');
                updateStageState(4, "pending");
                checkStageCompletion();
                document.getElementById("stageContent").innerHTML = `
                <div id="landingMessage" class="d-flex flex-column align-items-center justify-content-center my-5">
                    <span class="fa-stack fa-4x">
                        <i class="fad fa-square fa-stack-2x text-primary"></i>
                        <i class="fad fa-bolt fa-stack-1x fa-inverse"></i>
                    </span>
                    <h2 class="mt-3 text-primary">Electrical Inspection Report</h2>
                    <p class="lead text-muted">Select a stage from the checklist to begin your inspection.</p>
                </div>`;
            } else {
                showResponse("danger", data.message);
            }
        })
        .catch(error => {
            console.error("Save error:", error);
            showResponse("danger", "Failed to save data.");
        });
}


// Function to handle post-update actions
function handleReportUpdate() {
    // Example of what to do after the update
    // You can perform operations like enabling or disabling form fields, changing UI, etc.
    if (is_update) {
        // Logic when it is an update
        console.log("Handling update logic.");
    } else {
        // Logic when it's a new report
        console.log("Handling new report logic.");
    }
}


function markStageComplete(stage) {
    // Update the stage to 'complete' first.
    updateStageState(stage, 'complete');

    // Prepare form data to send to the backend
    let formData = new URLSearchParams();
    formData.append("stage", stage);
    formData.append("status", 'complete');
    formData.append("is_update", is_update); // Pass the update flag
    formData.append("master_report_id", MASTER_REPORT_ID); // Pass the master_report_id

    // Sending the data to the backend for update (replace with the appropriate URL)
    fetch(`${ROOT_URL}/includes/elec/update_stage_status.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: formData.toString()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Handle success
                console.log("Stage marked as complete.");
            } else {
                console.error("Error marking stage complete:", data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
        });

    // Close the modal and reset content
    killModal('confirmBox');
    document.getElementById("stageContent").innerHTML = `
        <div id="landingMessage" class="d-flex flex-column align-items-center justify-content-center my-5">
            <span class="fa-stack fa-4x">
                <i class="fad fa-square fa-stack-2x text-primary"></i>
                <i class="fad fa-bolt fa-stack-1x fa-inverse"></i>
            </span>
            <h2 class="mt-3 text-primary">Electrical Inspection Report</h2>
            <p class="lead text-muted">Select a stage from the checklist to begin your inspection.</p>
        </div>`;
}


function clearStage(stage) {
    let formData = new URLSearchParams();
    formData.append("master_report_id", MASTER_REPORT_ID);
    formData.append("stage", stage);
    formData.append("tenant", TENANT_URL);
    fetch(`${ROOT_URL}/includes/elec/clear_stage.php`, {
            method: "POST",
            headers: {
                "Content-Type": "application/x-www-form-urlencoded"
            },
            body: formData.toString()
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                killModal('confirmBox');
                updateStageState(stage, "empty");
                document.getElementById("stageContent").innerHTML = "<h5>Select a stage from the checklist to begin.</h5>";
            } else {
                console.error("Error clearing stage:", data.message);
            }
        })
        .catch(error => console.error("Error clearing stage:", error));
}

function saveReportDetails() {
    const form = document.getElementById('formComplete');
    const formData = new FormData(form);
    formData.append("stage", `save_stage_1`);
    formData.append("reportComplete", `1`);
    fetch(`${ROOT_URL}/includes/elec/load_stage.php?tenant=${TENANT_URL}`, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showResponse("success", "Installation details saved successfully.");
            } else {
                showResponse("danger", "Error saving installation details: " + data.message);
            }
        })
        .catch(error => {
            console.error('Error saving installation details:', error);
            alert('An error occurred while saving installation details.');
        });
}

function cancelReport() {
    alert("Cancel report functionality is triggered.");
}

var currentPage = 1;

function showCircuitPage(page) {
    var container = document.getElementById('circuitContainer');
    var direction = null;
    if (page > currentPage) {
        direction = 'right';
    } else if (page < currentPage) {
        direction = 'left';
    }

    if (direction) {
        // Animate the container sliding out.
        container.style.transition = 'transform 0.4s ease';
        // If moving to a higher page, slide left (i.e. content moves left).
        // If moving to a lower page, slide right.
        container.style.transform = (direction === 'right') ? 'translateX(-100%)' : 'translateX(100%)';

        // After the slide-out, update the visible cards and slide the container back in.
        setTimeout(function() {
            // Hide all cards.
            document.querySelectorAll('.circuit-card').forEach(function(card) {
                card.style.display = 'none';
            });
            // Show only the cards that belong to the requested page.
            document.querySelectorAll('.circuit-card[data-page="' + page + '"]').forEach(function(card) {
                card.style.display = 'block';

            });

            // Immediately reset the container position offscreen (from the opposite side).
            container.style.transition = 'none';
            container.style.transform = (direction === 'right') ? 'translateX(100%)' : 'translateX(-100%)';

            // Force a reflow so the browser registers the change.
            container.offsetHeight;

            // Animate the container sliding in to its normal position.
            container.style.transition = 'transform 0.4s ease';
            container.style.transform = 'translateX(0)';

            // Update the current page value.
            currentPage = page;
            initSlimForVisibleCircuits;
            // Update the pagination active state.
            document.querySelectorAll('.pagination .page-item').forEach(function(li) {
                li.classList.remove('active');
            });
            var activeLink = document.querySelector('.pagination .page-item a[onclick*="showCircuitPage(' + page + ')"]');
            if (activeLink && activeLink.parentElement) {
                activeLink.parentElement.classList.add('active');
            }
        }, 400); // Matches the slide-out duration (400ms)
    } else {
        // If no direction (i.e. page is the same), simply update the active pagination link.
        document.querySelectorAll('.pagination .page-item').forEach(function(li) {
            li.classList.remove('active');

        });
        var activeLink = document.querySelector('.pagination .page-item a[onclick*="showCircuitPage(' + page + ')"]');
        if (activeLink && activeLink.parentElement) {
            activeLink.parentElement.classList.add('active');
        }
    }
}

function addInstrumentRow() {
    let table = document.getElementById("instrumentRows");
    let rowCount = table.rows.length;
    let row = table.insertRow();

    row.innerHTML = '<td><input type="text" name="instrument[]" class="form-control" placeholder="Instrument"></td>' +
        '<td><input type="text" name="maker[]" class="form-control" placeholder="Maker"></td>' +
        '<td><input type="text" name="serial_no[]" class="form-control" placeholder="Serial No"></td>' +
        '<td><input type="date" name="last_calibration[]" class="form-control"></td>' +
        '<td><input type="date" name="next_calibration[]" class="form-control"></td>' +
        ' <td class="text-center">' +
        '<button type="button" class="btn btn-outline-danger btn-sm removeInstrumentRow">' +
        '<i class="fad fa-trash-alt"></i>' +
        ' </button>' +
        ' </td>';
    updateInstrumentRowIcons();
}
// Update instrument row icons (handle first row separately)
function updateInstrumentRowIcons() {
    let rows = document.querySelectorAll("#instrumentRows tr");
    rows.forEach((row, index) => {
        let actionCell = row.querySelector("td:last-child");
        if (!actionCell) {
            actionCell = document.createElement("td");
            actionCell.className = "text-center";
            row.appendChild(actionCell);
        }

        if (index === 0) {
            actionCell.innerHTML =
                '<button type="button" class="btn btn-outline-warning btn-sm clearInstrumentRow">' +
                '<i class="fad fa-broom"></i>' +
                '</button>';
        } else {
            actionCell.innerHTML =
                '<button type="button" class="btn btn-outline-danger btn-sm removeInstrumentRow">' +
                '<i class="fad fa-trash-alt"></i>' +
                '</button>';
        }
    });

    attachInstrumentRowEvents();
}

// Attach event listeners for dynamically added rows
function attachInstrumentRowEvents() {
    document.querySelectorAll(".removeInstrumentRow").forEach(button => {
        button.onclick = function() {
            this.closest("tr").remove();
            updateInstrumentRowIcons();
        };
    });

    document.querySelectorAll(".clearInstrumentRow").forEach(button => {
        button.onclick = function() {
            let row = this.closest("tr");
            row.querySelectorAll("input").forEach(input => input.value = "");
        };
    });
}
document.addEventListener("DOMContentLoaded", updateInstrumentRowIcons);
</script>
<?php if ($circuitsCount === 0): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var stageContent = document.getElementById('stageContent');
    if (stageContent) {
        stageContent.innerHTML =
            '<div id="landingMessage" class="d-flex flex-column align-items-center justify-content-center my-5">' +
            '<span class="fa-stack fa-4x">' +
            '<i class="fad fa-square fa-stack-2x text-dark"></i>' +
            '<i class="fad fa-bolt fa-stack-1x text-light fa-inverse"></i>' +
            '</span>' +
            '<h2 class="mt-3 text-primary">Electrical Inspection Report</h2>' +
            '<p class="lead text-muted">Select a stage from the checklist to begin your inspection.</p>' +
            '<div class="alert alert-warning mt-3">No circuits detected. Please <a href="index.php?t=includes/master&p=elec.php&view=elec_new_circuit&data=<?= $d[0]?>|<?= $d[1]?>|<?= $d[2]?>" class="alert-link">define at least one circuit</a>.' +
            '</div>' +
            '</div>';
    }
    var stageItems = document.querySelectorAll('#reportChecklist li');
    stageItems.forEach(function(li) {
        li.classList.add('disabled');
        li.style.pointerEvents = 'none';
        li.style.opacity = '0.6';
    });
});
</script>
<?php endif; ?>